//---------------------------------------------------------------------------
// Stochastic indicator
//---------------------------------------------------------------------------
library Stochastic;

uses
  SysUtils, classes, graphics, windows, IndicatorInterfaceUnit;

var
  // External variables
  KPeriod: integer;
  DPeriod: integer;
  Slowing: integer;
  ApplyTo: integer;

  // Buffers
  Kfast, Kslow, Dline: TIndexBuffer;


//---------------------------------------------------------------------------
// Initialize indicator
//---------------------------------------------------------------------------
procedure Init; stdcall;
begin
  // define properties
  IndicatorShortName('Stochastic');
  SetOutputWindow(ow_SeparateWindow);
  SetFixedMinMaxValues(0, 100);
  AddLevel(20, psDot, 1, cl_GridColor);
  AddLevel(80, psDot, 1, cl_GridColor);
  SetEmptyValue(105);

  // register options
  AddSeparator('Common');

  RegOption('%K period', ot_Integer, KPeriod);
  SetOptionRange('%K period', 1, MaxInt);
  KPeriod := 8;

  RegOption('%D period', ot_Integer, DPeriod);
  SetOptionRange('%D period', 1, MaxInt);
  DPeriod := 3;

  RegOption('Slowing', ot_Integer, Slowing);
  SetOptionRange('Slowing', 1, MaxInt);
  Slowing := 3;

  RegOption('Apply to', ot_EnumType, ApplyTo);
  AddOptionValue('Apply to', 'High/Low');
  AddOptionValue('Apply to', 'Close/Close');
  ApplyTo := 0;

  // create buffers
  Kfast := CreateIndexBuffer;
  Kslow := CreateIndexBuffer;
  Dline := CreateIndexBuffer;
  IndicatorBuffers(2);
  SetIndexBuffer(0, Kslow);
  SetIndexBuffer(1, Dline);
  SetIndexStyle(0, ds_line, psSolid, 1, RGB($1E, $90, $FF));
  SetIndexLabel(0, 'K Line');
  SetIndexStyle(1, ds_Line, psDot, 1, clRed);
  SetIndexLabel(1, 'D Line');
end;

//---------------------------------------------------------------------------
// Deinitialize indicator
//---------------------------------------------------------------------------
procedure Done; stdcall;
begin

end;

//---------------------------------------------------------------------------
// Calculate requested bar
//---------------------------------------------------------------------------
procedure Calculate(index: integer); stdcall;
var
  sum, Plow, Phigh: double;
  i: integer;

  function price(i: integer): double;
  begin
    if ApplyTo = 0 then
      result := (Low(i) + High(i))/2
    else
      result := Close(i);
  end;

begin
  if (Bars < KPeriod) or (Bars < Slowing) or (Bars < DPeriod) then
    exit;

  // find highest and lowest price for period KPeriod
  Phigh := High(index);
  Plow := Low(index);
  for i:=1 to KPeriod - 1 do
    begin
      if High(index + i) > Phigh then
        Phigh := High(index + i);
      if Low(index + i) < Plow then
        Plow := Low(index + i);
    end;

  // count Kfast line value
  if Phigh <= Plow then
    Kfast[index] := 50
  else
    Kfast[index] := 100*((price(index) - Plow)/(Phigh - Plow));

  // get Kslow
  sum := 0;
  for i:=0 to Slowing - 1 do
    sum := sum + Kfast[index + i];
  Kslow[index] := sum/Slowing;

  // get Dline
  sum := 0;
  for i:=0 to DPeriod - 1 do
    sum := sum + Kslow[index + i];
  Dline[index] := sum/DPeriod;
end;

exports

Init, Done, Calculate;

end.
