//---------------------------------------------------------------------------
// ADX
//---------------------------------------------------------------------------
library ADX;

uses
  SysUtils, Math, graphics, IndicatorInterfaceUnit, TechnicalFunctions;

var
  // External variables
  ADXperiod: integer;
  ApplyToPrice: integer;

  // Buffers
  ADXBuffer: TIndexBuffer;
  PlusDiBuffer: TIndexBuffer;
  MinusDiBuffer: TIndexBuffer;
  PlusSdiBuffer: TIndexBuffer;
  MinusSdiBuffer: TIndexBuffer;
  TempBuffer: TIndexBuffer;

//---------------------------------------------------------------------------
// Initialize indicator
//---------------------------------------------------------------------------
procedure Init; stdcall;
begin
  // define properties
  SetOutputWindow(ow_SeparateWindow);
  IndicatorShortName('Average Directional Movement Index (ADX)');

  // register options
  AddSeparator('Common');

  RegOption('Period', ot_Integer, ADXPeriod);
  SetOptionRange('Period', 1, MaxInt);
  ADXperiod := 14;

  RegApplyToPriceOption(ApplyToPrice);

  // create buffers
  ADXBuffer := CreateIndexBuffer;
  PlusDiBuffer := CreateIndexBuffer;
  MinusDiBuffer := CreateIndexBuffer;
  PlusSdiBuffer := CreateIndexBuffer;
  MinusSdiBuffer := CreateIndexBuffer;
  TempBuffer := CreateIndexBuffer;

  IndicatorBuffers(3);
  SetIndexBuffer(0,ADXBuffer);
  SetIndexLabel(0, 'ADX');
  SetIndexBuffer(1,PlusDiBuffer);
  SetIndexLabel(1, '+Di');
  SetIndexBuffer(2,MinusDiBuffer);
  SetIndexLabel(2, '-Di');

  SetIndexStyle(0, ds_Line, psSolid, 1, clYellow);
  SetIndexStyle(1, ds_Line, psSolid, 1, clGreen);
  SetIndexStyle(2, ds_Line, psSolid, 1, clBlue);
end;

//---------------------------------------------------------------------------
// Deinitialize indicator
//---------------------------------------------------------------------------
procedure Done; stdcall;
begin

end;

//---------------------------------------------------------------------------
// Calculate requested bar
//---------------------------------------------------------------------------
procedure Calculate(index: integer); stdcall;
var
  divide, pdm, mdm, tr, price_high, price_low, k: double;
  num1, num2, num3: double;
begin
  price_low := Low(index);
  price_high := High(index);

  //----
  pdm := price_high - High(index+1);
  mdm := Low(index+1) - price_low;
  if(pdm < 0) then
    pdm:=0;
  if(mdm < 0) then
    mdm:=0;
  if(pdm = mdm) then
    begin
      pdm:=0;
      mdm:=0;
    end
  else
    if(pdm < mdm) then
      pdm := 0
    else
      if(mdm < pdm) then
        mdm:=0;

  //---- calculate real interval
  num1 := Abs(price_high - price_low);
  num2 := Abs(price_high - GetPrice(index + 1, TPriceType(ApplyToPrice)));
  num3 := Abs(price_low - GetPrice(index + 1, TPriceType(ApplyToPrice)));
  tr := Max(num1, num2);
  tr := Max(tr, num3);

  //---- counting plus/minus direction
  if(tr = 0) then
    begin
      PlusSdiBuffer[index] := 0;
      MinusSdiBuffer[index]:=0;
    end
  else
    begin
      PlusSdiBuffer[index] := 100.0*pdm/tr;
      MinusSdiBuffer[index] := 100.0*mdm/tr;
    end;

  k := 2/(ADXperiod + 1);
  //---- apply EMA to +DI
  if (index = Bars - 2) then
    PlusDiBuffer[index + 1] := PlusSdiBuffer[index + 1];
  PlusDiBuffer[index] := PlusSdiBuffer[index]*k + PlusDiBuffer[index + 1]*(1 - k);

  //---- apply EMA to -DI
  if(index = Bars - 2) then
    MinusDiBuffer[index + 1] := MinusSdiBuffer[index + 1];
  MinusDiBuffer[index] := MinusSdiBuffer[index]*k + MinusDiBuffer[index + 1]*(1 - k);

  //---- Directional Movement (DX)
  divide := Abs(PlusDiBuffer[index] + MinusDiBuffer[index]);
  if (divide = 0) then
    TempBuffer[index] := 0
  else
    TempBuffer[index] := 100*(Abs(PlusDiBuffer[index] - MinusDiBuffer[index])/divide);

  //---- ADX is exponential moving average on DX
  if (index = Bars - 2) then
    ADXBuffer[index + 1] := TempBuffer[index+1];
  ADXBuffer[index] := TempBuffer[index]*k + ADXBuffer[index + 1]*(1 - k);
end;

exports

  Init, Done, Calculate;

end.
