//---------------------------------------------------------------------------
// Relative Strength Index indicator
//---------------------------------------------------------------------------
library RSI;

uses
  SysUtils, classes, graphics, windows, IndicatorInterfaceUnit;

var
  // External variables
  period: integer;
  ApplyToPrice: integer;

  // Buffers
  _RSI, AvGain, AvLoss: TIndexBuffer;


//---------------------------------------------------------------------------
// Initialize indicator
//---------------------------------------------------------------------------
procedure Init; stdcall;
begin
  // define properties
  IndicatorShortName('RSI');
  SetOutputWindow(ow_SeparateWindow);
  SetFixedMinMaxValues(0, 100);
  AddLevel(30, psDot, 1, cl_GridColor);
  AddLevel(70, psDot, 1, cl_GridColor);
  SetEmptyValue(105);

  // register options
  AddSeparator('Common');

  RegOption('Period', ot_Integer, period);
  SetOptionRange('Period', 1, MaxInt);
  period := 14;

  RegOption('Apply to price', ot_EnumType, ApplyToPrice);
  AddOptionValue('Apply to price', 'Close');
  AddOptionValue('Apply to price', 'Open');
  AddOptionValue('Apply to price', 'High');
  AddOptionValue('Apply to price', 'Low');
  AddOptionValue('Apply to price', '(High + Low)/2');
  AddOptionValue('Apply to price', '(High + Low + Close)/3');
  AddOptionValue('Apply to price', '(High + Low + Close + Close)/4');
  ApplyToPrice := 0;

  // create buffers
  _RSI := CreateIndexBuffer;
  AvGain := CreateIndexBuffer;
  AvLoss := CreateIndexBuffer;

  IndicatorBuffers(1);
  SetIndexBuffer(0, _RSI);
  SetIndexStyle(0, ds_Line, psSolid, 1, RGB($1E, $90, $FF));
  SetIndexLabel(0, 'RSI');
end;

//---------------------------------------------------------------------------
// Deinitialize indicator
//---------------------------------------------------------------------------
procedure Done; stdcall;
begin

end;

//---------------------------------------------------------------------------
// Calculate requested bar
//---------------------------------------------------------------------------
procedure Calculate(index: integer); stdcall;
var
  i: integer;
  diff, gain, loss: double;

  function GetPrice(index: integer): double;
  begin
    case ApplyToPrice of
      0:   result := Close(index);
      1:   result := Open(index);
      2:   result := High(index);
      3:   result := Low(index);
      4:   result := (High(index) + Low(index))/2;
      5:   result := (High(index) + Low(index) + Close(index))/3;
      6:   result := (High(index) + Low(index) + Close(index)*2)/4;
      else result := 0;
    end;
  end;

begin
  if Bars - index <= period then
    exit;

  if Bars - index = period + 1 then
    begin
      gain := 0;
      loss := 0;
      for i:=index to index + period - 1 do
        begin
          diff := GetPrice(i) - GetPrice(i + 1);
          if diff > 0 then
            gain := gain + diff
          else
            loss := loss - diff;
        end;
      AvGain[index] := gain/period;
      AvLoss[index] := loss/period;
    end
  else
    begin
      gain := 0;
      loss := 0;
      diff := GetPrice(index) - GetPrice(index + 1);
      if diff > 0 then
        gain := diff
      else
        loss := -diff;
      gain := (AvGain[index + 1]*(period - 1) + gain)/period;
      loss := (AvLoss[index + 1]*(period - 1) + loss)/period;
      AvGain[index] := gain;
      AvLoss[index] := loss;
      if loss = 0 then
        _RSI[index] := 105
      else
        _RSI[index] := 100 - 100/(1 + gain/loss);
    end;
end;

exports

Init, Done, Calculate;

end.
