//---------------------------------------------------------------------------
// MACD indicator
//---------------------------------------------------------------------------
library MACD;

uses
  SysUtils, classes, graphics, windows, IndicatorInterfaceUnit;

var
  // External variables
  FastEMAPeriod: integer;
  SlowEMAPeriod: integer;
  SMAPeriod: integer;
  ApplyToPrice: integer;

  // Buffers
  FastEMA, SlowEMA, _MACD, SMA: TIndexBuffer;


//---------------------------------------------------------------------------
// Initialize indicator
//---------------------------------------------------------------------------
procedure Init; stdcall;
begin
  // define properties
  IndicatorShortName('MACD');
  SetOutputWindow(ow_SeparateWindow);
  AddLevel(0, psDot, 1, cl_GridColor);

  // register options
  AddSeparator('Common');

  RegOption('Fast EMA period', ot_Integer, FastEMAPeriod);
  SetOptionRange('Fast EMA period', 1, MaxInt);
  FastEMAPeriod := 5;

  RegOption('Slow EMA period', ot_Integer, SlowEMAPeriod);
  SetOptionRange('Slow EMA period', 1, MaxInt);
  SlowEMAPeriod := 13;

  RegOption('SMA period', ot_Integer, SMAPeriod);
  SetOptionRange('SMA period', 1, MaxInt);
  SMAPeriod := 3;

  RegOption('Apply to price', ot_EnumType, ApplyToPrice);
  AddOptionValue('Apply to price', 'Close');
  AddOptionValue('Apply to price', 'Open');
  AddOptionValue('Apply to price', 'High');
  AddOptionValue('Apply to price', 'Low');
  AddOptionValue('Apply to price', '(High + Low)/2');
  AddOptionValue('Apply to price', '(High + Low + Close)/3');
  AddOptionValue('Apply to price', '(High + Low + Close + Close)/4');
  ApplyToPrice := 0;

  // create buffers
  FastEMA := CreateIndexBuffer;
  SlowEMA := CreateIndexBuffer;
  _MACD := CreateIndexBuffer;
  SMA := CreateIndexBuffer;

  IndicatorBuffers(2);
  SetIndexBuffer(0, _MACD);
  SetIndexBuffer(1, SMA);
  SetIndexStyle(0, ds_Histogram, psSolid, 1, clSilver);
  SetIndexLabel(0, 'MACD');
  SetIndexStyle(1, ds_Line, psDot, 1, clRed);
  SetIndexLabel(1, 'Signal Line');
end;

//---------------------------------------------------------------------------
// Deinitialize indicator
//---------------------------------------------------------------------------
procedure Done; stdcall;
begin

end;

//---------------------------------------------------------------------------
// Calculate requested bar
//---------------------------------------------------------------------------
procedure Calculate(index: integer); stdcall;
var
  i: integer;
  k, sum: double;

  function GetPrice(index: integer): double;
  begin
    case ApplyToPrice of
      0:   result := Close(index);
      1:   result := Open(index);
      2:   result := High(index);
      3:   result := Low(index);
      4:   result := (High(index) + Low(index))/2;
      5:   result := (High(index) + Low(index) + Close(index))/3;
      6:   result := (High(index) + Low(index) + Close(index)*2)/4;
      else result := 0;
    end;
  end;

begin
  // calculate fast EMA
  k := 2/(FastEMAPeriod + 1);
  if index = Bars - 1 then
    FastEMA[index] := GetPrice(index)
  else
    FastEMA[index] := FastEMA[index + 1] + k*(GetPrice(index) - FastEMA[index + 1]);

  // calculate slow EMA
  k := 2/(SlowEMAPeriod + 1);
  if index = Bars - 1 then
    SlowEMA[index] := GetPrice(index)
  else
    SlowEMA[index] := SlowEMA[index + 1] + k*(GetPrice(index) - SlowEMA[index + 1]);

  // calculate MACD
  _MACD[index] := FastEMA[index] - SlowEMA[index];

  // calculate SMA
  sum := 0;
  for i:=index to index + SMAPeriod - 1 do
    sum := sum + _MACD[i];
  SMA[index] := sum/SMAPeriod;
end;

exports

Init, Done, Calculate;

end.
