//---------------------------------------------------------------------------
// Bollinger bands indicator
//---------------------------------------------------------------------------
library BollingerBands;

uses
  graphics, IndicatorInterfaceUnit, TechnicalFunctions;

var
  // External variables
  period: integer = 8;
  Shift: integer = 0;
  Deviation: double = 2;
  ApplyToPrice: integer;

  // Buffers
  EMA, UpBand, DownBand: TIndexBuffer;


//---------------------------------------------------------------------------
// Initialize indicator
//---------------------------------------------------------------------------
procedure Init; stdcall;
begin
  // define properties
  IndicatorShortName('Bollinger Bands');
  SetOutputWindow(ow_ChartWindow);

  // register options
  AddSeparator('Common');

  RegOption('Period', ot_Integer, period);
  SetOptionRange('Period', 1, MaxInt);
  RegOption('Deviation', ot_Double, Deviation);
  SetOptionRange('Deviation', 0.1, 200);
  RegOption('Shift', ot_Integer, Shift);
  RegApplyToPriceOption(ApplyToPrice);

  // create buffers
  EMA := CreateIndexBuffer;
  UpBand := CreateIndexBuffer;
  DownBand := CreateIndexBuffer;

  IndicatorBuffers(3);

  SetIndexBuffer(0, UpBand);
  SetIndexStyle(0, ds_Line, psSolid, 1, clTeal);
  SetIndexLabel(0, 'Up band');

  SetIndexBuffer(1, EMA);
  SetIndexStyle(1, ds_Line, psSolid, 1, clTeal);
  SetIndexLabel(1, 'EMA');

  SetIndexBuffer(2, DownBand);
  SetIndexLabel(2, 'Down band');
  SetIndexStyle(2, ds_Line, psSolid, 1, clTeal);
end;

//---------------------------------------------------------------------------
// parameters changed
//---------------------------------------------------------------------------
procedure OnParamsChange; stdcall;
begin
  SetBufferShift(0, Shift);
  SetBufferShift(1, Shift);
  SetBufferShift(2, Shift);
end;

//---------------------------------------------------------------------------
// Calculate requested bar
//---------------------------------------------------------------------------
procedure Calculate(index: integer); stdcall;
var
  i: integer;
  sum, value, sd: double;

  function GetPrice(index: integer): double;
  begin
    result := TechnicalFunctions.GetPrice(index, TPriceType(ApplyToPrice));
  end;

begin
  if (index + period) >= Bars then
    exit;

  value := GetMA(index, 0, period, ma_EMA, TPriceType(ApplyToPrice), EMA[index + 1]);
  EMA[index] := value;

  // calculate bands
  if value <> 0 then
    begin
      sum := 0;
      for i:=index to index + period - 1 do
        sum := sum + sqr(abs(GetPrice(i) - value));
      sd := sqrt(sum/period)*Deviation;
      UpBand[index] := EMA[index] + sd;
      DownBand[index] := EMA[index] - sd;
    end
  else
    begin
      UpBand[index] := 0;
      DownBand[index] := 0;
    end;
end;

exports

Init, OnParamsChange, Calculate;

end.
